import bpy
from mathutils import Vector, Quaternion, Matrix, Euler
import math

from ..core import state

# Try to import Blender 5.0 animation utilities
try:
    from bpy_extras import anim_utils
    HAS_ANIM_UTILS = True
except:
    HAS_ANIM_UTILS = False


def convert_root(args):
    result_q = Quaternion((args[7], args[4], args[6], -args[5]))
    result_l = Vector((-args[1], -args[3], args[2]))
    return {"location": result_l, "rotation_quaternion": result_q}


def convert_pose(args):
    humanoid_bone = state.config.humanoid_to_bone[args[0]]
    parent_quaternion = humanoid_bone.parent_quaternion
    local_rotation = humanoid_bone.local_rotation
    
    incoming_rotation = Quaternion((args[7], args[4], args[6], -args[5]))
    result_l = Vector((0, 0, 0))

    if args[0] == "Hips":
        root_quaternion = state.config.humanoid_to_bone["root"].parent_quaternion
        result_l = root_quaternion.inverted(
        ) @ ((Vector((-args[1], -args[3], args[2])) - Vector((-0.0000, -0.0025, 0.8788))) * bpy.context.scene.dollars_hip_height / 0.8788)

    result_q: Quaternion = parent_quaternion.inverted() @ incoming_rotation @ parent_quaternion
    return {"location": result_l, "rotation_quaternion": result_q}


def create_fcurve_in_action(action: bpy.types.Action, data_path: str, array_index: int = 0,
                           slot_identifier: int = 0, action_group: str = ""):
    # Blender 4.5 and older (Legacy)
    if hasattr(action, 'fcurves'):
        try:
            if action_group:
                fcurve = action.fcurves.new(data_path=data_path, index=array_index, action_group=action_group)
            else:
                fcurve = action.fcurves.new(data_path=data_path, index=array_index)
            return fcurve
        except Exception as e:
            print(f"Error creating fcurve in Blender 4.x: {e}")
            return None

    # Blender 5.0+ (Slotted Actions)
    else:
        try:
            if not action.slots:
                action.slots.new(name="Slot_0", id_type="OBJECT")
            try:
                action_slot = action.slots[slot_identifier]
            except IndexError:
                action_slot = action.slots.new(name=f"Slot_{slot_identifier}", id_type="OBJECT")
            if not HAS_ANIM_UTILS:
                print("ERROR: bpy_extras.anim_utils not available for Blender 5.0")
                return None
                
            channelbag = anim_utils.action_ensure_channelbag_for_slot(action, action_slot)
            
            if not channelbag:
                print(f"ERROR: Failed to get channelbag for slot {slot_identifier}")
                return None
            if action_group:
                fcurve = channelbag.fcurves.ensure(data_path, index=array_index, group_name=action_group)
            else:
                fcurve = channelbag.fcurves.ensure(data_path, index=array_index)
            
            return fcurve
            
        except Exception as e:
            print(f"Error creating fcurve in Blender 5.0: {e}")
            import traceback
            traceback.print_exc()
            return None


def get_fcurves_from_action(action: bpy.types.Action, slot_identifier: int = 0):
    # Blender 4.x
    if hasattr(action, 'fcurves'):
        return list(action.fcurves)
    
    # Blender 5.0+
    else:
        try:
            if not action.slots or len(action.slots) <= slot_identifier:
                return []
            
            action_slot = action.slots[slot_identifier]
            
            if not HAS_ANIM_UTILS:
                return []
            
            channelbag = anim_utils.action_get_channelbag_for_slot(action, action_slot)
            
            if channelbag and hasattr(channelbag, 'fcurves'):
                return list(channelbag.fcurves)
            
            return []
        except Exception as e:
            print(f"Error getting fcurves: {e}")
            return []