from json import JSONDecoder
from typing import Dict

import bpy
from bpy import context
from bpy.props import StringProperty
from mathutils import Quaternion
from . import hip_calculator


class Config:
    decorder = JSONDecoder()

    def __init__(self):
        self.init_root()
        self.init_pose()

    def init_root(self):
        self.root = bpy.context.scene.dollars_target_armature

    def init_pose(self):
        self.humanoid_to_bone = {}
        result = {}
        armature = bpy.context.scene.dollars_target_armature
        
        if not armature:
            self.humanoid_to_bone = {}
            return
        
        human_body_bones = [
            *human_body_bones_torso,
            *human_body_bones_right_hand,
            *human_body_bones_left_hand
        ]
        
        for bone_name in human_body_bones:
            try:
                binded_bone_name = getattr(
                    bpy.context.scene,
                    "dollars_bones_" + bone_name
                )
                if binded_bone_name != "":
                    if binded_bone_name in armature.pose.bones:
                        result[bone_name] = HumanoidBone(
                            armature.pose.bones[binded_bone_name])
            except Exception as e:
                pass
                
        if result.get("Hips"):
            root_bone = result["Hips"].pose_bone
            while root_bone.parent:
                root_bone = root_bone.parent
            result["root"] = HumanoidBone(root_bone)

        self.humanoid_to_bone: Dict[str, HumanoidBone] = result

class HumanoidBone:
    def __init__(self, pose_bone: bpy.types.PoseBone):
        self.pose_bone: bpy.types.PoseBone = pose_bone

        self.local_rotation: Quaternion = pose_bone.rotation_quaternion.copy()

        quat = pose_bone.bone.matrix.to_quaternion()
        if not pose_bone.bone.use_inherit_rotation:
            self.parent_quaternion: Quaternion = quat
            return
        for parent in pose_bone.bone.parent_recursive:
            quat = parent.matrix.to_quaternion() @ quat
            if not parent.use_inherit_rotation:
                break
        self.parent_quaternion: Quaternion = quat
        
human_body_bones_torso = [
    "Hips",
    "Spine",
    "Chest",
    "UpperChest",
    "Neck",
    "Head",
    "LeftShoulder",
    "LeftUpperArm",
    "LeftLowerArm",
    "LeftHand",
    "LeftUpperLeg",
    "LeftLowerLeg",
    "LeftFoot",
    "LeftToes",
    "RightShoulder",
    "RightUpperArm",
    "RightLowerArm",
    "RightHand",
    "RightUpperLeg",
    "RightLowerLeg",
    "RightFoot",
    "RightToes"
]

human_body_bones_left_hand = [
    "LeftThumbProximal",
    "LeftThumbIntermediate",
    "LeftThumbDistal",
    "LeftIndexProximal",
    "LeftIndexIntermediate",
    "LeftIndexDistal",
    "LeftMiddleProximal",
    "LeftMiddleIntermediate",
    "LeftMiddleDistal",
    "LeftRingProximal",
    "LeftRingIntermediate",
    "LeftRingDistal",
    "LeftLittleProximal",
    "LeftLittleIntermediate",
    "LeftLittleDistal"
]

human_body_bones_right_hand = [
    "RightThumbProximal",
    "RightThumbIntermediate",
    "RightThumbDistal",
    "RightIndexProximal",
    "RightIndexIntermediate",
    "RightIndexDistal",
    "RightMiddleProximal",
    "RightMiddleIntermediate",
    "RightMiddleDistal",
    "RightRingProximal",
    "RightRingIntermediate",
    "RightRingDistal",
    "RightLittleProximal",
    "RightLittleIntermediate",
    "RightLittleDistal"
]

def register():
    setattr(bpy.types.Scene, "dollars_bones_Hips", StringProperty(
        name="Hips",
        description="Select the bone that corresponds to the humanoids Hips bone",
        update=hip_calculator.update_hip_bone
    ))
    
    for bone in human_body_bones_torso:
        if bone != "Hips":
            setattr(bpy.types.Scene, "dollars_bones_" + bone, StringProperty(
                name=bone,
                description="Select the bone that corresponds to the humanoids bone"
            ))
            
    for bone in human_body_bones_left_hand:
        setattr(bpy.types.Scene, "dollars_bones_" + bone, StringProperty(
            name=bone,
            description="Select the bone that corresponds to the humanoids bone"
        ))
        
    for bone in human_body_bones_right_hand:
        setattr(bpy.types.Scene, "dollars_bones_" + bone, StringProperty(
            name=bone,
            description="Select the bone that corresponds to the humanoids bone"
        ))


def unregister():
    for bone in human_body_bones_torso:
        delattr(bpy.types.Scene, "dollars_bones_" + bone)
        
    for bone in human_body_bones_left_hand:
        delattr(bpy.types.Scene, "dollars_bones_" + bone)
        
    for bone in human_body_bones_right_hand:
        delattr(bpy.types.Scene, "dollars_bones_" + bone)

