import bpy
import os
import json
from bpy.props import StringProperty
from bpy_extras.io_utils import ImportHelper, ExportHelper
from ..core import config

class ExportMappingsButton(bpy.types.Operator, ExportHelper):
    """Save current mapping configuration to JSON file"""
    bl_idname = "dollarsmocap.save_mappings"
    bl_label = "Save Mappings"
    
    filename_ext = ".json"
    filter_glob: StringProperty(default="*.json", options={'HIDDEN'})
    
    def execute(self, context):
        data = {}
        
        # Save blendshape mappings
        blendshape_data = {}
        for i, item in enumerate(context.scene.blendshape_remap_d):
            from ..panels.remapping import TARGET_BLENDSHAPES
            if i < len(TARGET_BLENDSHAPES) - 3:  # Excluding the last 3 items (HeadYaw, HeadPitch, HeadRoll)
                target = TARGET_BLENDSHAPES[i]
                source = item.source
                blendshape_data[target] = source
        
        data["blendshape_mappings"] = blendshape_data
        
        # Save bone mappings (including empty values)
        bone_data = {}
        
        # Save torso bones
        for bone in config.human_body_bones_torso:
            bone_value = getattr(context.scene, "dollars_bones_" + bone, "")
            bone_data[bone] = bone_value  # Save even if empty
        
        # Save left hand bones
        for bone in config.human_body_bones_left_hand:
            bone_value = getattr(context.scene, "dollars_bones_" + bone, "")
            bone_data[bone] = bone_value  # Save even if empty
        
        # Save right hand bones
        for bone in config.human_body_bones_right_hand:
            bone_value = getattr(context.scene, "dollars_bones_" + bone, "")
            bone_data[bone] = bone_value  # Save even if empty
        
        data["bone_mappings"] = bone_data
        
        # Write to file
        try:
            with open(self.filepath, 'w', encoding='utf-8') as f:
                json.dump(data, f, indent=4)
            self.report({'INFO'}, f"Successfully saved all mappings to {self.filepath}")
        except Exception as e:
            self.report({'ERROR'}, f"Error saving mappings: {str(e)}")
            return {'CANCELLED'}
        
        return {'FINISHED'}

    def invoke(self, context, event):
        self.filepath = "dollarsmocap_mappings.json"
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}


class ImportMappingsButton(bpy.types.Operator, ImportHelper):
    """Load mapping configuration from JSON file"""
    bl_idname = "dollarsmocap.load_mappings"
    bl_label = "Load Mappings"
    
    filename_ext = ".json"
    filter_glob: StringProperty(default="*.json", options={'HIDDEN'})
    
    def execute(self, context):
        try:
            with open(self.filepath, 'r', encoding='utf-8') as f:
                data = json.load(f)
        except Exception as e:
            self.report({'ERROR'}, f"Error loading file: {str(e)}")
            return {'CANCELLED'}
        
        # Load blendshape mappings if available
        if "blendshape_mappings" in data:
            blendshape_data = data["blendshape_mappings"]
            
            # Get the TARGET_BLENDSHAPES list from remapping.py
            from ..panels.remapping import TARGET_BLENDSHAPES
            
            # Make sure blendshape_remap_d is initialized
            if len(context.scene.blendshape_remap_d) != len(TARGET_BLENDSHAPES) - 3:
                context.scene.blendshape_remap_d.clear()
                for _ in range(len(TARGET_BLENDSHAPES) - 3):  # Excluding HeadYaw, HeadPitch, HeadRoll
                    context.scene.blendshape_remap_d.add()
            
            # Apply the loaded mappings
            for i, target in enumerate(TARGET_BLENDSHAPES[:-3]):  # Excluding HeadYaw, HeadPitch, HeadRoll
                if target in blendshape_data:
                    context.scene.blendshape_remap_d[i].source = blendshape_data[target]
        
        # Load bone mappings if available
        if "bone_mappings" in data:
            bone_data = data["bone_mappings"]
            armature_obj = context.scene.dollars_target_armature
            
            # Apply bone mappings
            for bone, value in bone_data.items():
                try:
                    if hasattr(context.scene, "dollars_bones_" + bone):
                        # Always set the value, even if it's empty
                        # If armature exists and value is not empty, verify the bone exists
                        if value and armature_obj and armature_obj.type == 'ARMATURE' and value not in armature_obj.pose.bones:
                            setattr(context.scene, "dollars_bones_" + bone, "")
                        else:
                            setattr(context.scene, "dollars_bones_" + bone, value)
                except Exception as e:
                    print(f"Error setting bone {bone}: {str(e)}")
        
        self.report({'INFO'}, f"Successfully loaded mappings from {self.filepath}")
        return {'FINISHED'}
